<?php
require_once "ai_client.php";

// Fetch latest active draw by status (1=daily, 2=bumper)
function getLatestDraw(mysqli $con, int $lotstatus): ?array {
    $stmt = mysqli_prepare($con, "SELECT lotid, lotname, lotdate, lottime FROM lottery WHERE lotstatus=? ORDER BY lotdate DESC, lottime DESC LIMIT 1");
    if (!$stmt) return null;
    mysqli_stmt_bind_param($stmt, "i", $lotstatus);
    mysqli_stmt_execute($stmt);
    $r = mysqli_stmt_get_result($stmt);
    $row = mysqli_fetch_assoc($r) ?: null;
    mysqli_free_result($r);
    mysqli_stmt_close($stmt);
    return $row;
}

// All available tickets for a draw (status=1)
function getActiveTickets(mysqli $con, int $lotid): array {
    $out = [];
    $stmt = mysqli_prepare($con, "SELECT ticketno FROM lotticket WHERE lotid=? AND status=1");
    if (!$stmt) return $out;
    mysqli_stmt_bind_param($stmt, "i", $lotid);
    mysqli_stmt_execute($stmt);
    $r = mysqli_stmt_get_result($stmt);
    while ($row = mysqli_fetch_assoc($r)) { $out[] = $row['ticketno']; }
    mysqli_free_result($r);
    mysqli_stmt_close($stmt);
    return $out;
}

// User’s historical tickets (optional signal)
function getUserBackupTickets(mysqli $con, int $userid): array {
    $out = [];
    $stmt = mysqli_prepare($con, "SELECT ticketno FROM backupbuyticket WHERE userid=? ORDER BY id DESC LIMIT 200");
    if (!$stmt) return $out;
    mysqli_stmt_bind_param($stmt, "i", $userid);
    mysqli_stmt_execute($stmt);
    $r = mysqli_stmt_get_result($stmt);
    while ($row = mysqli_fetch_assoc($r)) { $out[] = $row['ticketno']; }
    mysqli_free_result($r);
    mysqli_stmt_close($stmt);
    return $out;
}

// Previous draw results (if you store them)
function getPreviousResults(mysqli $con, int $limit = 50): array {
    // Adjust table/column names to your schema
    $out = [];
    if ($stmt = mysqli_prepare($con, "SELECT winning_number FROM results ORDER BY draw_date DESC LIMIT ?")) {
        mysqli_stmt_bind_param($stmt, "i", $limit);
        mysqli_stmt_execute($stmt);
        $r = mysqli_stmt_get_result($stmt);
        while ($row = mysqli_fetch_assoc($r)) { $out[] = $row['winning_number']; }
        mysqli_free_result($r);
        mysqli_stmt_close($stmt);
    }
    return $out;
}

// Build + call AI to suggest numbers that exist within available ticket pool
function aiLuckyNumbers(mysqli $con, int $userid, int $lotstatus, int $max = 11): ?array {
    // 1) Which draw?
    $draw = getLatestDraw($con, $lotstatus);
    if (!$draw) return null;

    $lotid   = (int)$draw['lotid'];
    $lotname = $draw['lotname'];
    $lotdate = $draw['lotdate'];
    $lottime = $draw['lottime'];

    // 2) Data to condition AI
    $availableTickets   = getActiveTickets($con, $lotid);               // pool to pick from
    $userBackupTickets  = getUserBackupTickets($con, $userid);          // user taste/history
    $previousResults    = getPreviousResults($con, 60);                  // optional prior outcomes

    // 3) Compact prompt (model picks ONLY from available tickets)
    $system = "You help a lottery site recommend interesting ticket numbers. 
Only suggest numbers that exactly exist in `available_tickets`. 
Do not invent or predict outcomes; this is for entertainment and pattern matching. 
Return a JSON array of up to {$max} strings. No extra text.";

    // Keep payloads light to control cost
    $user = [
        "lottery" => [
            "lotid"   => $lotid,
            "name"    => $lotname,
            "date"    => $lotdate,
            "time"    => $lottime
        ],
        "available_tickets_sample" => array_slice($availableTickets, 0, 1200), // cap to keep prompt small
        "user_history_sample"      => array_slice($userBackupTickets, 0, 200),
        "previous_results_sample"  => array_slice($previousResults, 0, 60),
        "max_recommendations"      => $max,
        // Simple heuristics for the model to prefer
        "preference_rules" => [
            "favor tickets sharing last-3 or last-4 digits with user_history",
            "favor numbers with repeated digits or common endings appearing in previous_results",
            "ensure diversity: avoid near-duplicates; mix endings",
            "must be present in available_tickets_sample"
        ]
    ];

    $messages = [
        ["role" => "system", "content" => $system],
        ["role" => "user",   "content" => json_encode($user, JSON_UNESCAPED_SLASHES)]
    ];

    $raw = openai_chat($messages, 'gpt-4o-mini', 250, 0.4);  // small+fast model per docs :contentReference[oaicite:3]{index=3}
    if (!$raw) return null;

    // The model returns a JSON array; make it robust if it adds extra text
    $json = null;
    // try a clean decode first
    $json = json_decode($raw, true);
    if (!is_array($json)) {
        // try to extract the first [ ... ] block
        if (preg_match('/\[[\s\S]*\]/', $raw, $m)) {
            $json = json_decode($m[0], true);
        }
    }
    if (!is_array($json)) return null;

    // Filter to only those that are actually available (safety)
    $availableSet = array_flip($availableTickets);
    $clean = [];
    foreach ($json as $t) {
        if (is_string($t) && isset($availableSet[$t])) {
            $clean[] = $t;
            if (count($clean) >= $max) break;
        }
    }
    return $clean;
}
